clear all
capture log close
program drop _all
set more off
snapshot erase _all

log using "..\Programs/Combining Measures of Globalization.log", replace


*** Combining Measures of Globalization.do
*** 8/1/2016
*** Brina Seidel

*************************************************************
*************************************************************
***  Combine data from all three flows
*************************************************************
*************************************************************

*************************************
*** Read in the data
*************************************
use "Global Migration - Percent of Population.dta", clear
merge 1:1 year using "Merchandise Exports - Percent of World GDP.dta", nogen
merge 1:1 year using "Foreign Capital Stock - Percent of Developing World GDP.dta", nogen 
drop if year == .

*************************************
*** Graph the data
*************************************
label variable foreign_cap_pct "Foreign Capital Stock (% Developing World GDP)"
label variable merch_exp_pct "Merchandise Exports (% Global GDP)"
label variable migrants_pct "Migrant Stock (% Global Population)"
format merch_exp_pct foreign_cap_pct %10.0f 
twoway (line merch_exp_pct foreign_cap_pct year if year >= 1870 & year <= 2015, yaxis(1) lwidth(thick thick)) ///
	(line migrants_pct year if year >= 1870 & year <= 1940, yaxis(2) lcolor("202 133 0") lwidth(thick)) (line migrants_pct year if year >= 1960 & year <= 2015, yaxis(2) lcolor("202 133 0") lwidth(thick)) ///
	(scatter foreign_cap_pct merch_exp_pct migrants_pct year if year==., msymbol(square square square) mcolor("192 0 26" "6 78 129" "202 133 0")), ///
	ytitle("% GDP", axis(1)) ytick(0(5)35, axis(1) tlength(0)) ylabel(5(5)35, noticks grid axis(1)) yscale(noline titlegap(2) axis(1)) ///
	ytitle("% Global Population", axis(2)) ytick(0(.5)3.5, axis(2) tlength(0)) ylabel(.5(.5)3.5, noticks grid axis(2)) yscale(noline titlegap(2) axis(2)) ///
	xtitle("Year",) xlabel(1870(10)2020, angle(45)) ///
	legend(cols(3) order(3 4 5) symysize(1) symxsize(1) size(small)) ///
	title("Globalization Trends, 1870-2015") plotregion(margin(zero))
graph display, xsize(10) ysize(5.5)
graph export "../Graphs/Trends in Globalization.png", as(png) replace 


*************************************
*** Save the data, and export the results for convenience 
*************************************
preserve
keep year migrants_pct merch_exp_pct foreign_cap_pct
keep if year >= 1870
save "Final Data for Trends in Globalization Graph.dta" ,replace
export excel using "..\Final Data for Trends in Globalization Graph.xlsx", replace firstrow(variables)
restore

*************************************************************
*************************************************************
*** Check peaks and when peaks were surpassed
*************************************************************
*************************************************************

*************************************
*** Fill in data for missing years
*************************************
keep if year >= 1870
tsset year
tsfill, full
foreach var of varlist foreign_cap_pct merch_exp_pct migrants_pct {

	rename `var' `var'_orig
	ipolate `var'_orig year, gen(`var')
	drop `var'_orig
		
}

*************************************
*** Get peaks and when they were surpassed
*************************************
foreach var of varlist foreign_cap_pct merch_exp_pct migrants_pct {

	*** Get value of Wave 1 peak
	gen wave1 = `var' if year <= 1950
	egen `var'_peak = max(wave1)
	label variable `var'_peak "Wave 1 Peak for `var'"
	drop wave1
	
	*** Get year of Wave 1 peak
	gen `var'_peakyr2 = year if round(`var', .001) == round(`var'_peak, .001) & year <= 1950
	egen `var'_peakyr = max(`var'_peakyr2)
	label variable `var'_peakyr "Year of Wave 1 Peak for `var'"
	drop `var'_peakyr2
	
	*** Get year when Wave 1 peak was surpassed
	gen `var'_surp = year if year >1950 & `var' > `var'_peak & `var' < .
	egen `var'_surpyr = min(`var'_surp)
	*** Check if this was done correctly
	format `var' `var'_peak %10.2fc
	list year `var' `var'_peak `var'_surpyr if year > 1950  & `var' < ., ab(20)
	drop `var'_surp
	label variable `var'_surpyr "Year When `var' Surpassed Wave 1 Peak"
}

*************************************
*** Show the results 
*************************************
rename foreign_cap_pct_* *1
rename merch_exp_pct_* *2
rename migrants_pct_* *3

preserve
keep year peak1-surpyr3
reshape long peak peakyr surpyr, i(year) j(measure)
label define measure 1 "Foreign Capital Stock" 2 "Merchandise Exports" 3 "Migrant Stock"
label values measure measure
drop year
duplicates drop
list, ab(30)
restore

log close
